﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace BLSpaceRay.Math3D
{
    public class RootSolver
    {
        #region " Methods "

        public static int SolveQuadratic(float _A, float _B, float _C, ref float[] T, int num)
        {
            float p, q, D;

            /* normal form: x^2 + px + q = 0 */

            p = _B / (2 * _A);
            q = _C / _A;

            D = p * p - q;

            if (MathHelper.IsZero(D))
            {
                T[num] = -p;
                return 1;
            }
            else if (D < 0)
            {
                return 0;
            }
            else if (D > 0)
            {
                float sqrt_D = MathHelper.Sqrt(D);

                T[num] = sqrt_D - p;
                T[num+1] = -sqrt_D - p;
                return 2;
            }
            return 0;
        }

        public static int SolveCubic(float _A, float _B, float _C, float _D, ref float[] T)
        {
            int i, num;
            float sub;
            float A, B, C;
            float sq_A, p, q;
            float cb_p, D;

            /* normal form: x^3 + Ax^2 + Bx + C = 0 */

            A = _B / _A;
            B = _C / _A;
            C = _D / _A;

            /*  substitute x = y - A/3 to eliminate quadric term:
            x^3 +px + q = 0 */

            sq_A = A * A;
            p = 1.0f / 3.0f * (-1.0f / 3.0f * sq_A + B);
            q = 1.0f / 2.0f * (2.0f / 27.0f * A * sq_A - 1.0f / 3.0f * A * B + C);

            /* use Cardano's formula */

            cb_p = p * p * p;
            D = q * q + cb_p;

            if (MathHelper.IsZero(D))
            {
                if (MathHelper.IsZero(q)) /* one triple solution */
                {
                    T[0] = 0.0f;
                    num = 1;
                }
                else /* one single and one double solution */
                {
                    float u = MathHelper.Cbrt(-q);
                    T[0] = 2.0f * u;
                    T[1] = -u;
                    num = 2;
                }
            }
            else if (D < 0) /* Casus irreducibilis: three real solutions */
            {
                float phi = 1.0f / 3.0f * MathHelper.Acos(-q / MathHelper.Sqrt(-cb_p));
                float t = 2.0f * MathHelper.Sqrt(-p);

                T[0] = t * MathHelper.Cos(phi);
                T[1] = -t * MathHelper.Cos(phi + MathHelper.PIOver3);
                T[2] = -t * MathHelper.Cos(phi - MathHelper.PIOver3);
                num = 3;
            }
            else /* one real solution */
            {
                float sqrt_D = MathHelper.Sqrt(D);
                float u = MathHelper.Cbrt(sqrt_D - q);
                float v = -MathHelper.Cbrt(sqrt_D + q);

                T[0] = u + v;
                num = 1;
            }

            /* resubstitute */

            sub = 1.0f / 3.0f * A;

            for (i = 0; i < num; ++i)
                T[i] -= sub;

            return num;
        }

        public static int SolveQuartic(float _A, float _B, float _C, float _D, float _E, ref float[] T)
        {
            float[]  coeffs = new float[4];
            float[] roots = new float[4];
            float  z, u, v, sub;
            float  A, B, C, D;
            float  sq_A, p, q, r;
            int     i, num;

            /* normal form: x^4 + Ax^3 + Bx^2 + Cx + D = 0 */

            A = _B / _A;
            B = _C / _A;
            C = _D / _A;
            D = _E / _A;

            /*  substitute x = y - A/4 to eliminate cubic term:
            x^4 + px^2 + qx + r = 0 */

            sq_A = A * A;
            p = -3.0f / 8.0f * sq_A + B;
            q = 1.0f / 8.0f * sq_A * A - 1.0f / 2.0f * A * B + C;
            r = -3.0f / 256.0f * sq_A * sq_A + 1.0f / 16.0f * sq_A * B - 1.0f / 4.0f * A * C + D;

            if (MathHelper.IsZero(r))
            {
                /* no absolute term: y(y^3 + py + q) = 0 */

                coeffs[0] = q;
                coeffs[1] = p;
                coeffs[2] = 0.0f;
                coeffs[3] = 1.0f;

                num = SolveCubic(coeffs[3], coeffs[2], coeffs[1], coeffs[0],ref T);

                roots[num++] = 0;
            }
            else
            {
                /* solve the resolvent cubic ... */

                coeffs[0] = (1.0f / 2.0f) * r * p - (1.0f / 8.0f) * q * q;
                coeffs[1] = -r;
                coeffs[2] = -1.0f / 2.0f * p;
                coeffs[3] = 1.0f;

                SolveCubic(coeffs[3], coeffs[2], coeffs[1], coeffs[0], ref T);

                /* ... and take the one real solution ... */

                z = T[ 0 ];

                /* ... to build two quadric equations */

                u = z * z - r;
                v = 2.0f * z - p;

                if (MathHelper.IsZero(u))
                    u = 0;
                else if (u > 0)
                    u = MathHelper.Sqrt(u);
                else
                    return 0;

                if (MathHelper.IsZero(v))
                    v = 0;
                else if (v > 0)
                    v = MathHelper.Sqrt(v);
                else
                    return 0;

                coeffs[0] = z - u;
                coeffs[1] = q < 0 ? -v : v;
                coeffs[2] = 1.0f;

                num = SolveQuadratic(coeffs[2], coeffs[1], coeffs[0], ref T, 0);

                coeffs[0] = z + u;
                coeffs[1] = q < 0 ? v : -v;
                coeffs[2] = 1.0f;

                num += SolveQuadratic(coeffs[2], coeffs[1], coeffs[0], ref T, num);
            }

            /* resubstitute */

            sub = 1.0f / 4.0f * A;

            for (i = 0; i < num; ++i)
                T[i] -= sub;

            return num;
        }

        #endregion
    }
}
